<?php

/**
 * MihanVideo Content Importer
 * A secure and optimized script for importing video content from MihanVideo sitemap
 * 
 * @author Improved Version
 * @version 2.0
 */

// Security and error handling
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

// Set execution limits
set_time_limit(900);
ini_set('max_execution_time', 900);
ini_set('memory_limit', '512M');

class MihanVideoImporter {
    
    private $sitemapUrl = "https://mihanvideo.com/sitemap-last.xml";
    private $maxVideos = 2000;
    private $minTitleLength = 40;
    private $wpdb;
    private $errors = [];
    private $processedCount = 0;
    private $skippedCount = 0;
    
    public function __construct() {
        // Initialize WordPress if not already loaded
        if (!function_exists('wp_insert_post')) {
            $wpLoadPath = '../wp-load.php';
            if (file_exists($wpLoadPath)) {
                require_once $wpLoadPath;
            } else {
                throw new Exception('WordPress not found. Please ensure wp-load.php is accessible.');
            }
        }
        
        global $wpdb;
        $this->wpdb = $wpdb;
    }
    
    /**
     * Main execution method
     */
    public function run() {
        try {
            echo "<h2>MihanVideo Content Importer Started</h2>\n";
            echo "<p>Processing up to {$this->maxVideos} videos...</p>\n";
            
            $sitemapContent = $this->fetchSitemap();
            if (!$sitemapContent) {
                throw new Exception('Failed to fetch sitemap content');
            }
            
            $videoUrls = $this->extractVideoUrls($sitemapContent);
            if (empty($videoUrls)) {
                throw new Exception('No video URLs found in sitemap');
            }
            
            $this->processVideos($videoUrls);
            
            $this->displayResults();
            
        } catch (Exception $e) {
            $this->logError('Fatal error: ' . $e->getMessage());
            echo "<p style='color: red;'>Error: " . htmlspecialchars($e->getMessage()) . "</p>\n";
        }
    }
    
    /**
     * Fetch sitemap content with proper error handling
     */
    private function fetchSitemap() {
        $ch = curl_init();
        
        if (!$ch) {
            throw new Exception('Failed to initialize cURL');
        }
        
        curl_setopt_array($ch, [
            CURLOPT_URL => $this->sitemapUrl,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2
        ]);
        
        $content = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        
        curl_close($ch);
        
        if ($content === false) {
            throw new Exception('cURL error: ' . $error);
        }
        
        if ($httpCode !== 200) {
            throw new Exception("HTTP error: {$httpCode}");
        }
        
        return $content;
    }
    
    /**
     * Extract video URLs from sitemap content
     */
    private function extractVideoUrls($content) {
        $urls = [];
        
        // Extract URL blocks
        if (preg_match_all('/(<url>)(.*?)(<\/url>)/is', $content, $matches)) {
            $urls = $matches[2];
        }
        
        return array_slice($urls, 0, $this->maxVideos);
    }
    
    /**
     * Process videos from URLs
     */
    private function processVideos($videoUrls) {
        foreach ($videoUrls as $index => $urlContent) {
            if ($index >= $this->maxVideos) {
                break;
            }
            
            try {
                $this->processSingleVideo($urlContent);
            } catch (Exception $e) {
                $this->logError("Error processing video {$index}: " . $e->getMessage());
                continue;
            }
        }
    }
    
    /**
     * Process a single video
     */
    private function processSingleVideo($urlContent) {
        $data = $this->extractVideoData($urlContent);
        
        if (!$this->validateVideoData($data)) {
            $this->skippedCount++;
            return;
        }
        
        // Check if video already exists
        if ($this->videoExists($data['title'])) {
            echo "<p style='color: orange;'>Skipped (duplicate): " . htmlspecialchars($data['title']) . "</p>\n";
            $this->skippedCount++;
            return;
        }
        
        // Create WordPress post
        $postId = $this->createWordPressPost($data);
        
        if ($postId) {
            $this->updateCustomFields($postId, $data);
            echo "<p style='color: green;'>Created: " . htmlspecialchars($data['title']) . " (ID: {$postId})</p>\n";
            $this->processedCount++;
        } else {
            $this->skippedCount++;
        }
    }
    
    /**
     * Extract video data from URL content
     */
    private function extractVideoData($content) {
        $data = [
            'url' => $this->extractField($content, '/(<loc>)(.*?)(<\/loc>)/i'),
            'title' => $this->extractField($content, '/(<video:title><!\[CDATA\[)(.*?)(\]\]>)/i'),
            'thumbnail' => $this->extractField($content, '/(<video:thumbnail_loc>)(.*?)(<\/video:thumbnail_loc>)/i'),
            'video' => $this->extractField($content, '/(<video:content_loc>)(.*?)(<\/video:content_loc>)/i'),
            'description' => $this->extractField($content, '/(<video:description><!\[CDATA\[)(.*?)(\]\]>)/i'),
            'tags' => $this->extractField($content, '/(<video:tag>)(.*?)(<\/video:tag><video:category)/i')
        ];
        
        // Clean up tags
        if (!empty($data['tags'])) {
            $data['tags'] = $this->cleanTags($data['tags']);
        }
        
        return $data;
    }
    
    /**
     * Extract field using regex
     */
    private function extractField($content, $pattern) {
        if (preg_match_all($pattern, $content, $matches)) {
            return $matches[2][0] ?? '';
        }
        return '';
    }
    
    /**
     * Clean and format tags
     */
    private function cleanTags($tags) {
        $tags = str_replace(['<video:tag> ', '<![CDATA[', '</video:tag>', ']]>'], ['', '', '', ','], $tags);
        return trim($tags, ',');
    }
    
    /**
     * Validate video data
     */
    private function validateVideoData($data) {
        if (empty($data['title']) || strlen($data['title']) < $this->minTitleLength) {
            return false;
        }
        
        if (empty($data['url']) || empty($data['video'])) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Check if video already exists
     */
    private function videoExists($title) {
        $title = $this->wpdb->esc_like($title);
        $count = $this->wpdb->get_var($this->wpdb->prepare(
            "SELECT COUNT(post_title) FROM {$this->wpdb->posts} WHERE post_title LIKE %s",
            $title
        ));
        
        return $count > 0;
    }
    
    /**
     * Create WordPress post
     */
    private function createWordPressPost($data) {
        // Sanitize title
        $title = function_exists('sanitize_text_field') ? 
            /** @phpstan-ignore-next-line */
            sanitize_text_field($data['title']) : 
            htmlspecialchars(strip_tags($data['title']), ENT_QUOTES, 'UTF-8');
        
        $postData = [
            'post_title' => $title,
            'post_content' => $this->generatePostContent($data),
            'post_status' => 'publish',
            'post_type' => 'post',
            'post_author' => 1
        ];
        
        if (!function_exists('wp_insert_post')) {
            $this->logError('WordPress function wp_insert_post not available');
            return false;
        }
        
        /** @phpstan-ignore-next-line */
        $postId = wp_insert_post($postData, true);
        
        if (function_exists('is_wp_error') && /** @phpstan-ignore-next-line */ is_wp_error($postId)) {
            $this->logError('Failed to create post: ' . $postId->get_error_message());
            return false;
        }
        
        return $postId;
    }
    
    /**
     * Generate post content
     */
    private function generatePostContent($data) {
        $thumbnailUrl = $this->processThumbnailUrl($data['thumbnail']);
        $tinyPic = str_replace('.jpg', '_200x130.jpg', $thumbnailUrl);
        
        // Escape URLs and attributes
        $escapedUrl = function_exists('esc_url') ? 
            /** @phpstan-ignore-next-line */
            esc_url($tinyPic) : 
            htmlspecialchars($tinyPic, ENT_QUOTES, 'UTF-8');
        
        $escapedTitle = function_exists('esc_attr') ? 
            /** @phpstan-ignore-next-line */
            esc_attr($data['title']) : 
            htmlspecialchars($data['title'], ENT_QUOTES, 'UTF-8');
        
        $escapedDescription = function_exists('wp_kses_post') ? 
            /** @phpstan-ignore-next-line */
            wp_kses_post($data['description']) : 
            htmlspecialchars($data['description'], ENT_QUOTES, 'UTF-8');
        
        return '<img class="nodisplay" src="' . $escapedUrl . '" title="' . $escapedTitle . '">' . 
               $escapedDescription;
    }
    
    /**
     * Process thumbnail URL for CDN
     */
    private function processThumbnailUrl($thumbnail) {
        if (empty($thumbnail)) {
            return '';
        }
        
        // Replace server prefixes
        $replacements = [
            '//s1' => '//ss1-',
            '//s2' => '//ss2-',
            '//s3' => '//ss3-',
            '//s4' => '//ss4-',
            '//s5' => '//ss5-',
            '//s6' => '//ss6-',
            '//s7' => '//ss7-',
            '//s8' => '//ss8-'
        ];
        
        foreach ($replacements as $search => $replace) {
            $thumbnail = str_replace($search, $replace, $thumbnail);
        }
        
        $thumbnail = str_replace('https://', '', $thumbnail);
        $thumbnail = str_replace('.mihanvideo.com/video_thumbs/', '', $thumbnail);
        
        return "https://cdn.arvandvideo.ir/s2_pic/" . $thumbnail;
    }
    
    /**
     * Update custom fields
     */
    private function updateCustomFields($postId, $data) {
        if (function_exists('update_field')) {
            // Escape URLs
            $videoUrl = function_exists('esc_url_raw') ? 
                /** @phpstan-ignore-next-line */
                esc_url_raw($data['video']) : 
                filter_var($data['video'], FILTER_SANITIZE_URL);
            
            $thumbnailUrl = function_exists('esc_url_raw') ? 
                /** @phpstan-ignore-next-line */
                esc_url_raw($data['thumbnail']) : 
                filter_var($data['thumbnail'], FILTER_SANITIZE_URL);
            
            $sourceUrl = function_exists('esc_url_raw') ? 
                /** @phpstan-ignore-next-line */
                esc_url_raw($data['url']) : 
                filter_var($data['url'], FILTER_SANITIZE_URL);
            
            /** @phpstan-ignore-next-line */
            update_field('s2_url', $videoUrl, $postId);
            /** @phpstan-ignore-next-line */
            update_field('s2_pic', $thumbnailUrl, $postId);
            /** @phpstan-ignore-next-line */
            update_field('s2_source', $sourceUrl, $postId);
        }
    }
    
    /**
     * Log error
     */
    private function logError($message) {
        $this->errors[] = $message;
        error_log('MihanVideo Importer: ' . $message);
    }
    
    /**
     * Display results
     */
    private function displayResults() {
        echo "<h3>Import Results:</h3>\n";
        echo "<p><strong>Processed:</strong> {$this->processedCount}</p>\n";
        echo "<p><strong>Skipped:</strong> {$this->skippedCount}</p>\n";
        echo "<p><strong>Errors:</strong> " . count($this->errors) . "</p>\n";
        
        if (!empty($this->errors)) {
            echo "<h4>Errors:</h4>\n";
            echo "<ul>\n";
            foreach ($this->errors as $error) {
                echo "<li style='color: red;'>" . htmlspecialchars($error) . "</li>\n";
            }
            echo "</ul>\n";
        }
    }
}

// Execute the importer
try {
    $importer = new MihanVideoImporter();
    $importer->run();
} catch (Exception $e) {
    echo "<p style='color: red;'>Fatal Error: " . htmlspecialchars($e->getMessage()) . "</p>\n";
    error_log('MihanVideo Importer Fatal Error: ' . $e->getMessage());
}

?>